require('dotenv').config();
const express = require('express');
const session = require('express-session');
const flash = require('connect-flash');
const path = require('path');
const { initializeDatabase, eventOps, imaginaryOps } = require('./config/database');

const app = express();
const START_PORT = Number.isFinite(Number(process.env.PORT)) ? Number(process.env.PORT) : 3000;

// Initialize database
initializeDatabase();

// View engine setup
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// Middleware
app.use(express.static(path.join(__dirname, 'public')));
app.use(express.urlencoded({ extended: true }));
app.use(express.json());

// Session configuration
app.use(session({
  secret: process.env.SESSION_SECRET || 'default_secret_key',
  resave: false,
  saveUninitialized: false,
  cookie: {
    secure: false, // Set to true in production with HTTPS
    maxAge: 24 * 60 * 60 * 1000 // 24 hours
  }
}));

// Flash messages
app.use(flash());

// Global variables middleware
app.use((req, res, next) => {
  res.locals.user = req.session.user || null;
  res.locals.isAdmin = req.session.user?.is_admin || false;
  res.locals.messages = {
    success: req.flash('success'),
    error: req.flash('error')
  };
  res.locals.whatsappNumber = process.env.WHATSAPP_BUSINESS_NUMBER;
  const rawSpin = parseInt(process.env.SPIN_DURATION || 8000);
  res.locals.spinDuration = rawSpin < 1000 ? rawSpin * 1000 : rawSpin;
  next();
});

// Routes
const indexRoutes = require('./routes/index');
const authRoutes = require('./routes/auth');
const userRoutes = require('./routes/user');
const adminRoutes = require('./routes/admin');
const apiRoutes = require('./routes/api');

app.use('/', indexRoutes);
app.use('/auth', authRoutes);
app.use('/user', userRoutes);
app.use('/admin', adminRoutes);
app.use('/api', apiRoutes);

const spinDurationRaw = parseInt(process.env.SPIN_DURATION || 8000);
const spinDurationMs = spinDurationRaw < 1000 ? spinDurationRaw * 1000 : spinDurationRaw;
const spinDurationSeconds = Math.max(1, Math.ceil(spinDurationMs / 1000));
const completingSinceExpr = `-${spinDurationSeconds} seconds`;

function toPromise(fn, ...args) {
  return new Promise((resolve, reject) => {
    fn(...args, (err, result) => {
      if (err) return reject(err);
      resolve(result);
    });
  });
}

async function schedulerTick() {
  try {
    const due = await toPromise(eventOps.getDueUpcoming);
    await Promise.all(
      (due || []).map(event =>
        toPromise(eventOps.setActive, event.id).catch(err => {
          console.error('Failed to activate event', event.id, err);
        })
      )
    );

    const toComplete = await toPromise(eventOps.getActiveToComplete, completingSinceExpr);

    for (const event of toComplete || []) {
      try {
        const realParticipants = await toPromise(imaginaryOps.getRealParticipants, event.id) || [];

        if (realParticipants.length > 0) {
          const winner = realParticipants[Math.floor(Math.random() * realParticipants.length)];
          await toPromise(eventOps.setWinnerReal, event.id, winner.user_id);
          continue;
        }

        let imaginaryParticipants = await toPromise(imaginaryOps.getImaginaryParticipants, event.id) || [];

        if (imaginaryParticipants.length === 0) {
          await toPromise(imaginaryOps.addRandomImaginaryParticipants, event.id);
          imaginaryParticipants = await toPromise(imaginaryOps.getImaginaryParticipants, event.id) || [];
        }

        if (imaginaryParticipants.length > 0) {
          const winner = imaginaryParticipants[Math.floor(Math.random() * imaginaryParticipants.length)];
          await toPromise(eventOps.setWinnerImaginary, event.id, winner.alias);
        } else {
          console.warn('No participants available to complete event', event.id);
        }
      } catch (eventError) {
        console.error('Error completing event', event.id, eventError);
      }
    }
  } catch (err) {
    console.error('Scheduler error:', err);
  }
}

setInterval(() => {
  schedulerTick().catch(err => {
    console.error('Scheduler tick failed:', err);
  });
}, 1000);

// 404 handler
app.use((req, res) => {
  res.status(404).render('error', { 
    title: 'Página Não Encontrada',
    message: 'A página que procura não existe.'
  });
});

// Error handler
app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).render('error', {
    title: 'Erro',
    message: 'Ocorreu um erro no servidor.'
  });
});

function listenOnNextAvailablePort(port) {
  const server = app.listen(port, () => {
    const actualPort = server.address().port;
    console.log(`🎰 Baicaa running on http://localhost:${actualPort}`);
  });

  server.on('error', (err) => {
    if (err && err.code === 'EADDRINUSE') {
      server.close(() => listenOnNextAvailablePort(port + 1));
      return;
    }

    console.error('Server error:', err);
    process.exit(1);
  });
}

listenOnNextAvailablePort(START_PORT);
