const sqlite3 = require('sqlite3').verbose();
const bcrypt = require('bcryptjs');
const crypto = require('crypto');
const path = require('path');

const dbPath = path.join(__dirname, '..', 'database.sqlite');
const db = new sqlite3.Database(dbPath);

// Pool of 10 Angolan names for imaginary participants
const ANGOLAN_ALIASES = [
  'António Manuel',
  'Maria_Joana',
  'José C4',
  'AnnaCristina',
  'João Francisco',
  'Isabel Luí22a',
  'Carlos Miguel_99',
  '98_Sofia',
  'Paul0 André',
  'Rosa Helena'
];

// Enable foreign keys
db.run('PRAGMA foreign_keys = ON');

function getCredentialsEncryptionKey() {
  const source = String(process.env.CREDENTIALS_ENCRYPTION_KEY || process.env.SESSION_SECRET || 'default_secret_key');
  return crypto.createHash('sha256').update(source).digest();
}

function encryptPasswordForDisplay(plainText) {
  if (!plainText) return null;
  const key = getCredentialsEncryptionKey();
  const iv = crypto.randomBytes(12);
  const cipher = crypto.createCipheriv('aes-256-gcm', key, iv);
  const encrypted = Buffer.concat([cipher.update(String(plainText), 'utf8'), cipher.final()]);
  const tag = cipher.getAuthTag();
  return `${iv.toString('base64')}.${tag.toString('base64')}.${encrypted.toString('base64')}`;
}

function decryptPasswordForDisplay(payload) {
  if (!payload) return null;
  try {
    const parts = String(payload).split('.');
    if (parts.length !== 3) return null;
    const [ivB64, tagB64, dataB64] = parts;
    const iv = Buffer.from(ivB64, 'base64');
    const tag = Buffer.from(tagB64, 'base64');
    const data = Buffer.from(dataB64, 'base64');
    const key = getCredentialsEncryptionKey();
    const decipher = crypto.createDecipheriv('aes-256-gcm', key, iv);
    decipher.setAuthTag(tag);
    const decrypted = Buffer.concat([decipher.update(data), decipher.final()]);
    return decrypted.toString('utf8');
  } catch (e) {
    return null;
  }
}

// Initialize database tables
function initializeDatabase() {
  db.serialize(() => {
    // Users table
    db.run(`
      CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        phone_number TEXT UNIQUE NOT NULL,
        password_hash TEXT NOT NULL,
        password_enc TEXT,
        alias TEXT,
        credits_balance INTEGER DEFAULT 0,
        is_admin INTEGER DEFAULT 0,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Add password_enc column if it doesn't exist (for existing databases)
    db.run(`ALTER TABLE users ADD COLUMN password_enc TEXT`, (err) => {
      // Column already exists, ignore
    });

    // Add alias column if it doesn't exist (for existing databases)
    db.run(`ALTER TABLE users ADD COLUMN alias TEXT`, (err) => {
      // Column already exists, ignore
    });

    // Add is_imaginary column if it doesn't exist (for existing databases)
    db.run(`ALTER TABLE participations ADD COLUMN is_imaginary INTEGER DEFAULT 0`, (err) => {
      // Column already exists, ignore
    });

    // Events table
    db.run(`
      CREATE TABLE IF NOT EXISTS events (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        event_datetime DATETIME NOT NULL,
        prize_type TEXT NOT NULL CHECK(prize_type IN ('amount', 'item')),
        prize_value TEXT NOT NULL,
        status TEXT DEFAULT 'upcoming' CHECK(status IN ('upcoming', 'active', 'completed')),
        winner_id INTEGER,
        winner_is_imaginary INTEGER DEFAULT 0,
        winner_alias TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (winner_id) REFERENCES users(id)
      )
    `);

    // Add imaginary winner columns if they don't exist (for existing databases)
    db.run(`ALTER TABLE events ADD COLUMN winner_is_imaginary INTEGER DEFAULT 0`, (err) => {
      // Column already exists, ignore
    });

    db.run(`ALTER TABLE events ADD COLUMN winner_alias TEXT`, (err) => {
      // Column already exists, ignore
    });

    // Normalize existing event_datetime values for consistent comparisons/parsing
    db.run(`
      UPDATE events
      SET event_datetime = REPLACE(event_datetime, 'T', ' ')
      WHERE INSTR(event_datetime, 'T') > 0
    `, (err) => {
      // Ignore
    });

    db.run(`
      UPDATE events
      SET event_datetime = event_datetime || ':00'
      WHERE LENGTH(event_datetime) = 16
      AND SUBSTR(event_datetime, 11, 1) = ' '
    `, (err) => {
      // Ignore
    });

    // Participations table
    db.run(`
      CREATE TABLE IF NOT EXISTS participations (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        event_id INTEGER NOT NULL,
        credits_committed INTEGER NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        is_imaginary INTEGER DEFAULT 0,
        FOREIGN KEY (user_id) REFERENCES users(id),
        FOREIGN KEY (event_id) REFERENCES events(id),
        UNIQUE(user_id, event_id)
      )
    `);

    // Imaginary participants table
    db.run(`
      CREATE TABLE IF NOT EXISTS imaginary_participants (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        event_id INTEGER NOT NULL,
        alias TEXT NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (event_id) REFERENCES events(id)
      )
    `);

    // Approved phone numbers table
    db.run(`
      CREATE TABLE IF NOT EXISTS approved_phones (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        phone_number TEXT UNIQUE NOT NULL,
        is_active INTEGER DEFAULT 1,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Create default admin if not exists
    db.get('SELECT id FROM users WHERE is_admin = 1', (err, adminExists) => {
      const adminUsername = String(process.env.ADMIN_USERNAME || 'admin').trim().replace(/\s+/g, '');
      const adminPassword = String(process.env.ADMIN_PASSWORD || 'admin123');
      if (!adminExists) {
        const hashedPassword = bcrypt.hashSync(adminPassword, 10);
        db.run(`
          INSERT INTO users (phone_number, password_hash, password_enc, credits_balance, is_admin)
          VALUES (?, ?, ?, ?, ?)
        `, [adminUsername, hashedPassword, encryptPasswordForDisplay(adminPassword), 0, 1]);
        console.log('Admin user created');
      } else {
        db.all('SELECT id, password_hash FROM users WHERE is_admin = 1', (err, admins) => {
          if (!err && admins) {
            admins.forEach((admin) => {
              if (!bcrypt.compareSync(adminPassword, admin.password_hash)) {
                const hashedPassword = bcrypt.hashSync(adminPassword, 10);
                db.run('UPDATE users SET password_hash = ?, password_enc = ? WHERE id = ?', 
                  [hashedPassword, encryptPasswordForDisplay(adminPassword), admin.id]);
              }
            });
          }
        });
      }
    });

    // Add some default approved phone numbers if none exist
    db.get('SELECT COUNT(*) as count FROM approved_phones', (err, result) => {
      if (!err && result && result.count === 0) {
        const defaultPhones = [
          '923456789',
          '924567890', 
          '925678901',
          '926789012',
          '927890123'
        ];
        
        defaultPhones.forEach(phone => {
          db.run('INSERT INTO approved_phones (phone_number) VALUES (?)', [phone]);
        });
        console.log('Default approved phone numbers added');
      }
    });

    console.log('Database initialized successfully');
  });
}

// User operations
const userOps = {
  create: (phoneNumber, password, alias = null, callback) => {
    const hashedPassword = bcrypt.hashSync(password, 10);
    db.run(`
      INSERT INTO users (phone_number, password_hash, password_enc, alias, credits_balance, is_admin)
      VALUES (?, ?, ?, ?, 0, 0)
    `, [phoneNumber, hashedPassword, encryptPasswordForDisplay(password), alias], callback);
  },

  updatePassword: (userId, newPassword, callback) => {
    const hashedPassword = bcrypt.hashSync(newPassword, 10);
    db.run('UPDATE users SET password_hash = ?, password_enc = ? WHERE id = ?', 
      [hashedPassword, encryptPasswordForDisplay(newPassword), userId], callback);
  },

  getDisplayPassword: (user) => {
    if (!user) return null;
    return decryptPasswordForDisplay(user.password_enc);
  },

  updateAlias: (userId, alias, callback) => {
    db.run('UPDATE users SET alias = ? WHERE id = ?', [alias, userId], callback);
  },

  findByPhone: (phoneNumber, callback) => {
    db.get('SELECT * FROM users WHERE phone_number = ?', [phoneNumber], callback);
  },

  findById: (id, callback) => {
    db.get('SELECT * FROM users WHERE id = ?', [id], callback);
  },

  verifyPassword: (user, password) => {
    return bcrypt.compareSync(password, user.password_hash);
  },

  updateCredits: (userId, amount, callback) => {
    db.run('UPDATE users SET credits_balance = credits_balance + ? WHERE id = ?', [amount, userId], callback);
  },

  setCredits: (userId, amount, callback) => {
    db.run('UPDATE users SET credits_balance = ? WHERE id = ?', [amount, userId], callback);
  },

  getAll: (callback) => {
    db.all('SELECT id, phone_number, alias, credits_balance, created_at, password_enc FROM users WHERE is_admin = 0', callback);
  }
};

// Event operations
const eventOps = {
  normalizeEventDatetime: (eventDatetime) => {
    if (!eventDatetime) return eventDatetime;
    let dt = String(eventDatetime).trim();
    dt = dt.replace('T', ' ');
    if (/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}$/.test(dt)) {
      dt = `${dt}:00`;
    }
    return dt;
  },

  create: (name, eventDatetime, prizeType, prizeValue, callback) => {
    const normalizedDatetime = eventOps.normalizeEventDatetime(eventDatetime);
    db.run(`
      INSERT INTO events (name, event_datetime, prize_type, prize_value, status)
      VALUES (?, ?, ?, ?, 'upcoming')
    `, [name, normalizedDatetime, prizeType, prizeValue], callback);
  },

  findById: (id, callback) => {
    db.get('SELECT * FROM events WHERE id = ?', [id], callback);
  },

  getUpcoming: (limit = 2, callback) => {
    db.all(`
      SELECT * FROM events 
      WHERE status IN ('upcoming', 'active') 
      AND datetime(event_datetime) >= datetime('now', 'localtime', '-1 hour')
      ORDER BY event_datetime ASC 
      LIMIT ?
    `, [limit], callback);
  },

  getAll: (callback) => {
    db.all('SELECT * FROM events ORDER BY event_datetime DESC', callback);
  },

  getAllUpcoming: (callback) => {
    db.all(`
      SELECT * FROM events 
      WHERE status IN ('upcoming', 'active')
      ORDER BY event_datetime ASC
    `, callback);
  },

  getDueUpcoming: (callback) => {
    db.all(`
      SELECT * FROM events
      WHERE status = 'upcoming'
      AND datetime(event_datetime) <= datetime('now', 'localtime')
      ORDER BY event_datetime ASC
    `, callback);
  },

  getActiveToComplete: (sinceExpr, callback) => {
    db.all(`
      SELECT * FROM events
      WHERE status = 'active'
      AND datetime(event_datetime) <= datetime('now', 'localtime', ?)
      ORDER BY event_datetime ASC
    `, [sinceExpr], callback);
  },

  getCompleted: (callback) => {
    db.all(`
      SELECT e.*, 
             u.phone_number as winner_phone, 
             u.alias as winner_user_alias,
             e.winner_alias as winner_imaginary_alias
      FROM events e 
      LEFT JOIN users u ON e.winner_id = u.id 
      WHERE e.status = 'completed' 
      ORDER BY e.event_datetime DESC
    `, callback);
  },

  update: (id, name, eventDatetime, prizeType, prizeValue, callback) => {
    const normalizedDatetime = eventOps.normalizeEventDatetime(eventDatetime);
    db.run(`
      UPDATE events 
      SET name = ?, event_datetime = ?, prize_type = ?, prize_value = ?
      WHERE id = ?
    `, [name, normalizedDatetime, prizeType, prizeValue, id], callback);
  },

  delete: (id, callback) => {
    db.serialize(() => {
      db.run('DELETE FROM participations WHERE event_id = ?', [id]);
      db.run('DELETE FROM events WHERE id = ?', [id], callback);
    });
  },

  setWinnerReal: (eventId, winnerId, callback) => {
    db.run(`
      UPDATE events
      SET winner_id = ?, winner_is_imaginary = 0, winner_alias = NULL, status = 'completed'
      WHERE id = ?
    `, [winnerId, eventId], callback);
  },

  setWinnerImaginary: (eventId, winnerAlias, callback) => {
    db.run(`
      UPDATE events
      SET winner_id = NULL, winner_is_imaginary = 1, winner_alias = ?, status = 'completed'
      WHERE id = ?
    `, [winnerAlias, eventId], callback);
  },

  // Backwards compatible: treat numeric winnerId as real
  setWinner: (eventId, winnerId, callback) => {
    eventOps.setWinnerReal(eventId, winnerId, callback);
  },

  setActive: (eventId, callback) => {
    db.run(`UPDATE events SET status = 'active' WHERE id = ?`, [eventId], callback);
  },

  getParticipantCount: (eventId, callback) => {
    db.get('SELECT COUNT(*) as count FROM participations WHERE event_id = ?', [eventId], (err, result) => {
      callback(err, result ? result.count : 0);
    });
  }
};

// Participation operations
const participationOps = {
  create: (userId, eventId, creditsCommitted, callback) => {
    db.run(`
      INSERT INTO participations (user_id, event_id, credits_committed)
      VALUES (?, ?, ?)
    `, [userId, eventId, creditsCommitted], callback);
  },

  findByUserAndEvent: (userId, eventId, callback) => {
    db.get(`
      SELECT * FROM participations WHERE user_id = ? AND event_id = ?
    `, [userId, eventId], callback);
  },

  findByUser: (userId, callback) => {
    db.all(`
      SELECT p.*, e.name as event_name, e.event_datetime, e.prize_type, e.prize_value, e.status
      FROM participations p
      JOIN events e ON p.event_id = e.id
      WHERE p.user_id = ?
      ORDER BY e.event_datetime ASC
    `, [userId], callback);
  },

  findByEvent: (eventId, callback) => {
    db.all(`
      SELECT p.*, u.phone_number, u.alias
      FROM participations p
      JOIN users u ON p.user_id = u.id
      WHERE p.event_id = ?
    `, [eventId], callback);
  },

  delete: (userId, eventId, callback) => {
    db.get('SELECT credits_committed FROM participations WHERE user_id = ? AND event_id = ?', [userId, eventId], (err, participation) => {
      if (err) return callback(err);
      if (participation) {
        db.run('DELETE FROM participations WHERE user_id = ? AND event_id = ?', [userId, eventId], (err) => {
          callback(err, participation.credits_committed);
        });
      } else {
        callback(null, 0);
      }
    });
  },

  getRandomParticipant: (eventId, callback) => {
    db.all(`
      SELECT p.*, u.phone_number, u.alias
      FROM participations p
      JOIN users u ON p.user_id = u.id
      WHERE p.event_id = ? AND p.is_imaginary = 0
      ORDER BY RANDOM()
      LIMIT 1
    `, [eventId], (err, participants) => {
      callback(err, participants && participants.length > 0 ? participants[0] : null);
    });
  }
};

// Imaginary participant operations
const imaginaryOps = {
  // Add 3 random imaginary participants to an event
  addRandomImaginaryParticipants: (eventId, callback) => {
    const safeCallback = typeof callback === 'function' ? callback : () => {};
    // Get existing imaginary participants for this event
    db.all('SELECT alias FROM imaginary_participants WHERE event_id = ?', [eventId], (err, existing) => {
      if (err) return safeCallback(err);
      
      const existingAliases = existing.map(e => e.alias);
      
      // Get available aliases (not already used for this event)
      const availableAliases = ANGOLAN_ALIASES.filter(alias => !existingAliases.includes(alias));
      
      // Add up to 3 random imaginary participants
      const toAdd = Math.min(3, availableAliases.length);
      const shuffled = [...availableAliases].sort(() => 0.5 - Math.random());
      
      let added = 0;
      const addNext = () => {
        if (added >= toAdd) {
          return safeCallback(null, toAdd);
        }
        
        const alias = shuffled[added];
        db.run('INSERT INTO imaginary_participants (event_id, alias) VALUES (?, ?)', [eventId, alias], (err) => {
          if (err) return safeCallback(err);
          added++;
          addNext();
        });
      };
      
      addNext();
    });
  },

  // Get all participants (real + imaginary) for display
  getAllParticipants: (eventId, callback) => {
    db.all(`
      SELECT p.id, p.user_id, u.phone_number, u.alias, p.is_imaginary
      FROM participations p
      JOIN users u ON p.user_id = u.id
      WHERE p.event_id = ?
    `, [eventId], (err, realParticipants) => {
      if (err) return callback(err);
      
      db.all(`
        SELECT id, alias, 1 as is_imaginary, NULL as user_id, NULL as phone_number
        FROM imaginary_participants
        WHERE event_id = ?
      `, [eventId], (err, imaginaryParticipants) => {
        if (err) return callback(err);
        
        callback(null, [...realParticipants, ...imaginaryParticipants]);
      });
    });
  },

  // Get imaginary participants only
  getImaginaryParticipants: (eventId, callback) => {
    db.all(`
      SELECT id, alias
      FROM imaginary_participants
      WHERE event_id = ?
    `, [eventId], callback);
  },

  // Get only real participants for winner selection
  getRealParticipants: (eventId, callback) => {
    db.all(`
      SELECT p.*, u.phone_number, u.alias
      FROM participations p
      JOIN users u ON p.user_id = u.id
      WHERE p.event_id = ? AND p.is_imaginary = 0
    `, [eventId], callback);
  },

  // Clear imaginary participants for an event
  clearImaginaryParticipants: (eventId, callback) => {
    db.run('DELETE FROM imaginary_participants WHERE event_id = ?', [eventId], callback);
  }
};

// Approved phone operations
const approvedPhoneOps = {
  // Add approved phone number
  add: (phoneNumber, callback) => {
    db.run('INSERT INTO approved_phones (phone_number) VALUES (?)', [phoneNumber], callback);
  },

  // Check if phone is approved
  isApproved: (phoneNumber, callback) => {
    db.get('SELECT id FROM approved_phones WHERE phone_number = ? AND is_active = 1', [phoneNumber], (err, result) => {
      callback(err, !!result);
    });
  },

  // Get all approved phones
  getAll: (callback) => {
    db.all('SELECT * FROM approved_phones ORDER BY created_at DESC', callback);
  },

  // Remove approved phone
  remove: (phoneId, callback) => {
    db.run('DELETE FROM approved_phones WHERE id = ?', [phoneId], callback);
  },

  // Toggle active status
  toggleActive: (phoneId, callback) => {
    db.get('SELECT is_active FROM approved_phones WHERE id = ?', [phoneId], (err, phone) => {
      if (err) return callback(err);
      if (phone) {
        const newStatus = phone.is_active ? 0 : 1;
        db.run('UPDATE approved_phones SET is_active = ? WHERE id = ?', [newStatus, phoneId], callback);
      } else {
        callback(null, null);
      }
    });
  }
};

module.exports = {
  db,
  initializeDatabase,
  userOps,
  eventOps,
  participationOps,
  imaginaryOps,
  approvedPhoneOps,
  ANGOLAN_ALIASES
};
