const express = require('express');
const router = express.Router();
const { isAdmin } = require('../middleware/auth');
const { userOps, eventOps, participationOps, imaginaryOps, approvedPhoneOps } = require('../config/database');

// Generate random password
function generatePassword(length = 8) {
  const chars = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghjkmnpqrstuvwxyz23456789';
  let password = '';
  for (let i = 0; i < length; i++) {
    password += chars.charAt(Math.floor(Math.random() * chars.length));
  }
  return password;
}

// Admin dashboard
router.get('/', isAdmin, (req, res) => {
  // First get all users
  userOps.getAll((err, rawUsers) => {
    if (err) {
      console.error('Error fetching users:', err);
      req.flash('error', 'Erro ao carregar usuários');
      return res.redirect('/');
    }

    const users = rawUsers.map(u => {
      const displayPass = userOps.getDisplayPassword(u);
      return {
        ...u,
        display_password: displayPass
      };
    });

    // Then get upcoming events
    eventOps.getAllUpcoming((err, upcomingEvents) => {
      if (err) {
        console.error('Error fetching upcoming events:', err);
        req.flash('error', 'Erro ao carregar eventos');
        return res.redirect('/');
      }

      // Then get completed events
      eventOps.getCompleted((err, completedEvents) => {
        if (err) {
          console.error('Error fetching completed events:', err);
          req.flash('error', 'Erro ao carregar eventos');
          return res.redirect('/');
        }

        // Finally get approved phones
        approvedPhoneOps.getAll((err, approvedPhones) => {
          if (err) {
            console.error('Error fetching approved phones:', err);
            req.flash('error', 'Erro ao carregar números aprovados');
            return res.redirect('/');
          }

          // Render the dashboard with all data
          res.render('admin/dashboard', {
            title: 'Painel Administrativo',
            users,
            upcomingEvents: upcomingEvents || [],
            completedEvents: completedEvents || [],
            approvedPhones: approvedPhones || []
          });
        });
      });
    });
  });
});

// Create new user
router.post('/create-user', isAdmin, (req, res, next) => {
  const rawPhoneNumber = req.body?.phone_number;
  const { alias } = req.body;
  const phone_number = String(rawPhoneNumber || '').trim().replace(/\s+/g, '');

  // Check if user already exists
  const existingUser = userOps.findByPhone(phone_number);
  if (existingUser) {
    req.flash('error', 'Este número já está registrado.');
    return res.redirect('/admin');
  }

  // Generate password
  const password = generatePassword();
  const trimmedAlias = alias ? alias.trim().substring(0, 30) : null;

  try {
    userOps.create(phone_number, password, trimmedAlias);
    
    // Store credentials temporarily for display
    req.session.newUserCredentials = {
      phone_number,
      password,
      alias: trimmedAlias
    };

    req.flash('success', 'Usuário criado com sucesso!');
  } catch (error) {
    console.error('Error creating user:', error);
    req.flash('error', 'Erro ao criar usuário.');
  }

  res.redirect('/admin');
});

// Reset user password
router.post('/reset-user-password', isAdmin, (req, res, next) => {
  const rawPhoneNumber = req.body?.phone_number;
  const phone_number = String(rawPhoneNumber || '').trim().replace(/\s+/g, '');

  const user = userOps.findByPhone(phone_number);
  if (!user || user.is_admin) {
    req.flash('error', 'Usuário não encontrado.');
    return res.redirect('/admin');
  }

  const password = generatePassword();

  try {
    userOps.updatePassword(user.id, password);

    req.session.newUserCredentials = {
      phone_number: user.phone_number,
      password,
      alias: user.alias
    };

    req.flash('success', 'Senha atualizada com sucesso!');
  } catch (error) {
    console.error('Error resetting user password:', error);
    req.flash('error', 'Erro ao atualizar senha.');
  }

  res.redirect('/admin');
});

// Get new user credentials (for AJAX)
router.get('/new-user-credentials', isAdmin, (req, res, next) => {
  const credentials = req.session.newUserCredentials;
  req.session.newUserCredentials = null;
  res.json(credentials || null);
});

// Recharge user credits
router.post('/recharge', isAdmin, (req, res, next) => {
  const { phone_number, amount } = req.body;
  const creditsAmount = parseInt(amount);

  const user = userOps.findByPhone(phone_number);
  if (!user) {
    req.flash('error', 'Usuário não encontrado.');
    return res.redirect('/admin');
  }

  userOps.updateCredits(user.id, creditsAmount);
  req.flash('success', `${creditsAmount} Kz adicionados à conta de ${phone_number}.`);
  res.redirect('/admin');
});

// Create event
router.post('/create-event', isAdmin, (req, res) => {
  const { name, event_datetime, prize_type, prize_value } = req.body;

  eventOps.create(name, event_datetime, prize_type, prize_value, (err, result) => {
    if (err) {
      console.error('Error creating event:', err);
      req.flash('error', 'Erro ao criar evento.');
      return res.redirect('/admin');
    }

    // Add 3 random imaginary participants to the new event
    const eventId = this.lastID || result?.lastInsertRowid;
    if (eventId) {
      imaginaryOps.addRandomImaginaryParticipants(eventId);
    }
    
    req.flash('success', 'Evento criado com sucesso!');
    res.redirect('/admin');
  });
});

// Update event
router.post('/update-event', isAdmin, (req, res, next) => {
  const { event_id, name, event_datetime, prize_type, prize_value } = req.body;

  try {
    eventOps.update(event_id, name, event_datetime, prize_type, prize_value);
    req.flash('success', 'Evento atualizado com sucesso!');
  } catch (error) {
    console.error('Error updating event:', error);
    req.flash('error', 'Erro ao atualizar evento.');
  }

  res.redirect('/admin');
});

// Delete event
router.post('/delete-event', isAdmin, (req, res) => {
  const { event_id } = req.body;

  participationOps.findByEvent(event_id, (err, participants) => {
    if (err) {
      console.error('Error fetching participants for delete:', err);
      req.flash('error', 'Não foi possível carregar participantes do evento.');
      return res.redirect('/admin');
    }

    const safeParticipants = Array.isArray(participants) ? participants : [];

    const refundPromises = safeParticipants.map((p) => {
      return new Promise((resolve, reject) => {
        userOps.updateCredits(p.user_id, p.credits_committed, (updateErr) => {
          if (updateErr) return reject(updateErr);
          resolve();
        });
      });
    });

    Promise.all(refundPromises)
      .catch((refundErr) => {
        console.error('Error refunding participants:', refundErr);
        throw refundErr;
      })
      .then(() => {
        eventOps.delete(event_id, (deleteErr) => {
          if (deleteErr) {
            console.error('Error deleting event:', deleteErr);
            req.flash('error', 'Erro ao eliminar evento.');
            return res.redirect('/admin');
          }

          req.flash('success', safeParticipants.length > 0
            ? 'Evento eliminado. Créditos devolvidos aos participantes.'
            : 'Evento eliminado.');
          res.redirect('/admin');
        });
      })
      .catch(() => {
        req.flash('error', 'Erro ao devolver créditos aos participantes.');
        res.redirect('/admin');
      });
  });
});

// Run roulette (select winner)
router.post('/run-roulette', isAdmin, (req, res, next) => {
  const { event_id } = req.body;

  const event = eventOps.findById(event_id);
  if (!event) {
    req.flash('error', 'Evento não encontrado.');
    return res.redirect('/admin');
  }

  // Prefer real participants for the roulette list, but allow imaginary if none
  const realParticipants = imaginaryOps.getRealParticipants(event_id);
  const imaginaryParticipants = imaginaryOps.getImaginaryParticipants(event_id);

  if (realParticipants.length === 0 && imaginaryParticipants.length === 0) {
    req.flash('error', 'Nenhum participante neste evento.');
    return res.redirect('/admin');
  }

  // Set event to active (for animation on home page)
  eventOps.setActive(event_id);

  // Store participants for display/animation
  req.session.activeRoulette = {
    event_id,
    participants: (realParticipants.length > 0 ? realParticipants : imaginaryParticipants).map(p => ({
      user_id: p.user_id || null,
      phone_number: p.phone_number || null,
      alias: p.alias
    }))
  };

  req.flash('success', 'Roleta iniciada! Vá para a página principal para ver.');
  res.redirect('/admin');
});

// Complete roulette (after animation)
router.post('/complete-roulette', isAdmin, (req, res, next) => {
  const { event_id, winner_id } = req.body;

  try {
    eventOps.setWinner(event_id, winner_id);
    req.session.activeRoulette = null;
    res.json({ success: true });
  } catch (error) {
    console.error('Error completing roulette:', error);
    res.json({ success: false, error: error.message });
  }
});

// Add approved phone number
router.post('/add-approved-phone', isAdmin, (req, res, next) => {
  const { phone_number } = req.body;

  if (!phone_number || phone_number.length !== 9) {
    req.flash('error', 'Número de telefone inválido.');
    return res.redirect('/admin');
  }

  try {
    approvedPhoneOps.add(phone_number);
    req.flash('success', `Número ${phone_number} adicionado à lista de aprovados.`);
  } catch (error) {
    req.flash('error', 'Este número já está na lista de aprovados.');
  }

  res.redirect('/admin');
});

// Remove approved phone number
router.post('/remove-approved-phone', isAdmin, (req, res, next) => {
  const { phone_id } = req.body;

  try {
    approvedPhoneOps.remove(phone_id);
    req.flash('success', 'Número removido da lista de aprovados.');
  } catch (error) {
    req.flash('error', 'Erro ao remover número.');
  }

  res.redirect('/admin');
});

// Toggle approved phone status
router.post('/toggle-approved-phone', isAdmin, (req, res, next) => {
  const { phone_id } = req.body;

  try {
    const result = approvedPhoneOps.toggleActive(phone_id);
    if (result) {
      const status = result.changes[0].newStatus === 1 ? 'ativado' : 'desativado';
      req.flash('success', `Número ${status} com sucesso.`);
    }
  } catch (error) {
    req.flash('error', 'Erro ao alterar status do número.');
  }

  res.redirect('/admin');
});

module.exports = router;
