const express = require('express');
const router = express.Router();
const { userOps, eventOps, participationOps, imaginaryOps } = require('../config/database');

// Anonymize phone number (show only last 4 digits)
function anonymizePhone(phone) {
  if (!phone || phone.length < 4) return '***-****';
  const last4 = phone.slice(-4);
  return `***-***-${last4}`;
}

// Get current event data
router.get('/current-event', (req, res) => {
  eventOps.getUpcoming(2, (err, events) => {
    if (err) {
      return res.json({ hasEvent: false });
    }

    if (!events || events.length === 0) {
      return res.json({ hasEvent: false });
    }

    const currentEvent = events[0];
    imaginaryOps.getAllParticipants(currentEvent.id, (err2, participants) => {
      const safeParticipants = Array.isArray(participants) ? participants : [];

      res.json({
        hasEvent: true,
        event: {
          id: currentEvent.id,
          name: currentEvent.name,
          datetime: currentEvent.event_datetime,
          prizeType: currentEvent.prize_type,
          prizeValue: currentEvent.prize_value,
          status: currentEvent.status,
          participantCount: safeParticipants.length,
          winnerId: currentEvent.winner_id,
          winnerIsImaginary: !!currentEvent.winner_is_imaginary,
          winnerAlias: currentEvent.winner_alias || null
        },
        nextEvent: events[1]
          ? {
              id: events[1].id,
              name: events[1].name,
              datetime: events[1].event_datetime
            }
          : null
      });
    });
  });
});

// Get participants for spinning animation (using alias)
router.get('/event/:id/participants', (req, res) => {
  const eventId = req.params.id;
  
  // Get all participants (real + imaginary)
  imaginaryOps.getAllParticipants(eventId, (err, allParticipants) => {
    if (err) {
      return res.json([]);
    }

    const safeAll = Array.isArray(allParticipants) ? allParticipants : [];
    const displayParticipants = safeAll.map((p) => {
      if (p.is_imaginary) {
        return {
          id: `imaginary_${p.id}`,
          alias: p.alias,
          display: p.alias
        };
      }

      return {
        id: p.user_id,
        alias: p.alias,
        display: p.alias || anonymizePhone(p.phone_number)
      };
    });

    res.json(displayParticipants);
  });
});

// Start roulette spin (admin only)
router.post('/start-spin', (req, res) => {
  if (!req.session.user || !req.session.user.is_admin) {
    return res.status(403).json({ error: 'Não autorizado' });
  }

  const { event_id } = req.body;

  eventOps.findById(event_id, (err, event) => {
    if (err || !event) {
      return res.status(404).json({ error: 'Evento não encontrado' });
    }

    imaginaryOps.getRealParticipants(event_id, (err2, realParticipants) => {
      if (err2) {
        return res.status(500).json({ error: 'Erro ao carregar participantes' });
      }

      imaginaryOps.getImaginaryParticipants(event_id, (err3, imaginaryParticipants) => {
        if (err3) {
          return res.status(500).json({ error: 'Erro ao carregar participantes' });
        }

        const safeReal = Array.isArray(realParticipants) ? realParticipants : [];
        const safeImag = Array.isArray(imaginaryParticipants) ? imaginaryParticipants : [];

        if (safeReal.length === 0 && safeImag.length === 0) {
          return res.status(400).json({ error: 'Nenhum participante' });
        }

        let winner;
        if (safeReal.length > 0) {
          const winnerIndex = Math.floor(Math.random() * safeReal.length);
          winner = { ...safeReal[winnerIndex], is_imaginary: 0 };
        } else {
          const winnerIndex = Math.floor(Math.random() * safeImag.length);
          winner = { ...safeImag[winnerIndex], is_imaginary: 1 };
        }

        // Set event to active
        eventOps.setActive(event_id, () => {
          // ignore
        });

        // Get all participants (real + imaginary) for display
        imaginaryOps.getAllParticipants(event_id, (err4, allParticipants) => {
          const safeAll = Array.isArray(allParticipants) ? allParticipants : [];

          res.json({
            success: true,
            winner: {
              id: winner.is_imaginary ? `imaginary_${winner.id}` : winner.user_id,
              is_imaginary: !!winner.is_imaginary,
              phone: winner.is_imaginary ? null : winner.phone_number,
              phoneAnonymized: winner.is_imaginary ? null : anonymizePhone(winner.phone_number),
              alias: winner.alias,
              display: winner.is_imaginary ? winner.alias : (winner.alias || anonymizePhone(winner.phone_number))
            },
            participants: safeAll.map((p) => {
              if (p.is_imaginary) {
                return {
                  id: `imaginary_${p.id}`,
                  alias: p.alias,
                  display: p.alias
                };
              }

              return {
                id: p.user_id,
                alias: p.alias,
                display: p.alias || anonymizePhone(p.phone_number)
              };
            })
          });
        });
      });
    });
  });
});

// Complete spin and set winner
router.post('/complete-spin', (req, res) => {
  if (!req.session.user || !req.session.user.is_admin) {
    return res.status(403).json({ error: 'Não autorizado' });
  }

  const { event_id, winner_id, winner_alias, winner_is_imaginary } = req.body;

  try {
    const isImaginary = winner_is_imaginary === true || winner_is_imaginary === 'true' || winner_is_imaginary === 1 || winner_is_imaginary === '1';
    if (isImaginary) {
      if (!winner_alias || String(winner_alias).trim().length === 0) {
        return res.status(400).json({ error: 'winner_alias é obrigatório para vencedor imaginário' });
      }
      eventOps.setWinnerImaginary(event_id, winner_alias, (err) => {
        if (err) return res.status(500).json({ error: 'Erro ao definir vencedor' });
        res.json({ success: true });
      });
    } else {
      if (!winner_id) {
        return res.status(400).json({ error: 'winner_id é obrigatório para vencedor real' });
      }
      eventOps.setWinnerReal(event_id, winner_id, (err) => {
        if (err) return res.status(500).json({ error: 'Erro ao definir vencedor' });
        res.json({ success: true });
      });
    }
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Check if current user won
router.get('/event/:id/my-result', (req, res) => {
  const eventId = req.params.id;

  eventOps.findById(eventId, (err, event) => {
    if (err || !event || event.status !== 'completed') {
      return res.json({ completed: false });
    }

    const isLoggedIn = !!req.session.user;
    const isImaginaryWinner = !!event.winner_is_imaginary;
    const isWinner = !isImaginaryWinner && isLoggedIn && event.winner_id === req.session.user.id;

    let winnerPhone = null;
    let winnerAlias = null;
    let winnerDisplay = null;

    if (isImaginaryWinner) {
      winnerPhone = null;
      winnerAlias = event.winner_alias;
      winnerDisplay = event.winner_alias;
      return res.json({
        completed: true,
        isWinner,
        isImaginaryWinner,
        winnerPhone,
        winnerAlias,
        winnerDisplay,
        prize: {
          type: event.prize_type,
          value: event.prize_value
        }
      });
    }

    if (!event.winner_id) {
      return res.json({
        completed: true,
        isWinner,
        isImaginaryWinner,
        winnerPhone,
        winnerAlias,
        winnerDisplay,
        prize: {
          type: event.prize_type,
          value: event.prize_value
        }
      });
    }

    userOps.findById(event.winner_id, (err2, winner) => {
      if (!err2 && winner) {
        winnerPhone = isWinner ? winner.phone_number : anonymizePhone(winner.phone_number);
        winnerAlias = winner.alias;
        winnerDisplay = winner.alias || (isWinner ? winner.phone_number : anonymizePhone(winner.phone_number));
      }

      res.json({
        completed: true,
        isWinner,
        isImaginaryWinner,
        winnerPhone,
        winnerAlias,
        winnerDisplay,
        prize: {
          type: event.prize_type,
          value: event.prize_value
        }
      });
    });
  });
});

// Get user balance (for logged in users)
router.get('/balance', (req, res) => {
  if (!req.session.user) {
    return res.status(401).json({ error: 'Não autenticado' });
  }

  userOps.findById(req.session.user.id, (err, user) => {
    if (err || !user) {
      return res.status(500).json({ error: 'Erro ao carregar saldo' });
    }
    res.json({ balance: user.credits_balance });
  });
});

module.exports = router;
