const express = require('express');
const router = express.Router();
const { isRegularUser } = require('../middleware/auth');
const { userOps, eventOps, participationOps, imaginaryOps } = require('../config/database');

// User dashboard
router.get('/', isRegularUser, (req, res) => {
  // Refresh user data from database
  const user = userOps.findById(req.session.user.id);
  const displayPassword = userOps.getDisplayPassword(user);
  req.session.user.credits_balance = user.credits_balance;
  req.session.user.alias = user.alias;
  req.session.user.phone_number = user.phone_number;

  // Get all upcoming events
  const events = eventOps.getAllUpcoming();
  
  // Get user's participations
  const userParticipations = participationOps.findByUser(req.session.user.id);
  const participationMap = {};
  userParticipations.forEach(p => {
    participationMap[p.event_id] = p.credits_committed;
  });

  // Combine events with participation info and correct participant counts
  const eventsWithParticipation = events.map(event => {
    const allParticipants = imaginaryOps.getAllParticipants(event.id);
    return {
      ...event,
      userCredits: participationMap[event.id] || 0,
      isEntered: !!participationMap[event.id],
      participantCount: allParticipants.length
    };
  });

  res.render('user/dashboard', {
    title: 'Minha Conta',
    user: {
      ...req.session.user,
      display_password: displayPassword
    },
    events: eventsWithParticipation
  });
});

// Update alias
router.post('/update-alias', isRegularUser, (req, res) => {
  const { alias } = req.body;
  const userId = req.session.user.id;

  const trimmedAlias = alias ? alias.trim().substring(0, 30) : null;
  
  userOps.updateAlias(userId, trimmedAlias);
  req.session.user.alias = trimmedAlias;

  req.flash('success', trimmedAlias ? `Apelido atualizado para "${trimmedAlias}".` : 'Apelido removido.');
  res.redirect('/user');
});

// Enter event
router.post('/enter-event', isRegularUser, (req, res) => {
  const { event_id, credits } = req.body;
  const userId = req.session.user.id;
  const creditsAmount = parseInt(credits);

  // Validate
  if (!creditsAmount || creditsAmount <= 0) {
    req.flash('error', 'Valor de créditos inválido.');
    return res.redirect('/user');
  }

  // Get fresh user data
  const user = userOps.findById(userId);
  
  if (creditsAmount > user.credits_balance) {
    req.flash('error', 'Saldo insuficiente.');
    return res.redirect('/user');
  }

  // Check if already entered
  const existingParticipation = participationOps.findByUserAndEvent(userId, event_id);
  if (existingParticipation) {
    req.flash('error', 'Você já está inscrito neste evento.');
    return res.redirect('/user');
  }

  // Check if event exists and is upcoming
  const event = eventOps.findById(event_id);
  if (!event || event.status === 'completed') {
    req.flash('error', 'Evento não disponível.');
    return res.redirect('/user');
  }

  // Deduct credits and create participation
  userOps.updateCredits(userId, -creditsAmount);
  participationOps.create(userId, event_id, creditsAmount);

  // Update session
  req.session.user.credits_balance -= creditsAmount;

  req.flash('success', `Inscrito com sucesso! ${creditsAmount} Kz debitados.`);
  res.redirect('/user');
});

// Leave event
router.post('/leave-event', isRegularUser, (req, res) => {
  const { event_id } = req.body;
  const userId = req.session.user.id;

  // Check event status
  const event = eventOps.findById(event_id);
  if (!event || event.status === 'completed' || event.status === 'active') {
    req.flash('error', 'Não é possível sair deste evento.');
    return res.redirect('/user');
  }

  // Delete participation and refund
  const refundedCredits = participationOps.delete(userId, event_id);
  
  if (refundedCredits > 0) {
    userOps.updateCredits(userId, refundedCredits);
    req.session.user.credits_balance += refundedCredits;
    req.flash('success', `Saiu do evento. ${refundedCredits} Kz devolvidos.`);
  } else {
    req.flash('error', 'Você não está inscrito neste evento.');
  }

  res.redirect('/user');
});

module.exports = router;
