require('dotenv').config();
const express = require('express');
const session = require('express-session');
const path = require('path');
const sqlite3 = require('sqlite3').verbose();
const bcrypt = require('bcryptjs');

const app = express();
const START_PORT = Number.isFinite(Number(process.env.PORT)) ? Number(process.env.PORT) : 3000;

// Database Setup
const db = new sqlite3.Database('roulette.db');

// Initialize Database Tables
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      phone_number TEXT UNIQUE NOT NULL,
      password_hash TEXT NOT NULL,
      credits_balance INTEGER DEFAULT 0,
      is_admin INTEGER DEFAULT 0,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS events (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      event_datetime DATETIME NOT NULL,
      prize_type TEXT NOT NULL,
      prize_value TEXT NOT NULL,
      status TEXT DEFAULT 'upcoming',
      winner_id INTEGER,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (winner_id) REFERENCES users(id)
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS participations (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      event_id INTEGER NOT NULL,
      credits_committed INTEGER NOT NULL,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (user_id) REFERENCES users(id),
      FOREIGN KEY (event_id) REFERENCES events(id),
      UNIQUE(user_id, event_id)
    )
  `);

  // Create default admin if not exists
  db.get('SELECT id FROM users WHERE is_admin = 1', (err, adminExists) => {
    if (!adminExists) {
      const hashedPassword = bcrypt.hashSync(process.env.ADMIN_PASSWORD || 'admin123', 10);
      db.run('INSERT INTO users (phone_number, password_hash, is_admin, credits_balance) VALUES (?, ?, 1, 0)',
        [process.env.ADMIN_USERNAME || 'admin', hashedPassword]);
      console.log('Admin user created');
    }
  });
});

// Middleware
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));
app.use(express.static(path.join(__dirname, 'public')));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(session({
  secret: process.env.SESSION_SECRET || 'default_secret',
  resave: false,
  saveUninitialized: false,
  cookie: { secure: false, maxAge: 24 * 60 * 60 * 1000 }
}));

// Make session and config available to all views
app.use((req, res, next) => {
  res.locals.user = req.session.user || null;
  res.locals.whatsappNumber = process.env.WHATSAPP_BUSINESS_NUMBER;
  res.locals.spinDuration = process.env.SPIN_DURATION_SECONDS || 8;
  next();
});

// Helper Functions
function anonymizePhone(phone) {
  if (!phone || phone.length < 4) return '***-***-****';
  return '***-***-' + phone.slice(-4);
}

function generatePassword(length = 8) {
  const chars = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghjkmnpqrstuvwxyz23456789';
  let password = '';
  for (let i = 0; i < length; i++) {
    password += chars.charAt(Math.floor(Math.random() * chars.length));
  }
  return password;
}

// Auth Middleware
function requireAuth(req, res, next) {
  if (!req.session.user) {
    return res.redirect('/?login=required');
  }
  next();
}

function requireAdmin(req, res, next) {
  if (!req.session.user || !req.session.user.is_admin) {
    return res.redirect('/?admin=required');
  }
  next();
}

// ============== ROUTES ==============

// HOME PAGE
app.get('/', (req, res) => {
  db.all(`
    SELECT e.*, 
           (SELECT COUNT(*) FROM participations WHERE event_id = e.id) as participant_count
    FROM events e 
    WHERE e.status = 'upcoming' AND e.event_datetime > datetime('now')
    ORDER BY e.event_datetime ASC 
    LIMIT 2
  `, (err, events) => {
    if (err) return res.status(500).send('Database error');

    db.get(`
      SELECT e.*, u.phone_number as winner_phone,
             (SELECT COUNT(*) FROM participations WHERE event_id = e.id) as participant_count
      FROM events e
      LEFT JOIN users u ON e.winner_id = u.id
      WHERE e.status = 'completed'
      ORDER BY e.event_datetime DESC
      LIMIT 1
    `, (err, lastEvent) => {
      if (err) return res.status(500).send('Database error');

      let participants = [];
      if (events.length > 0) {
        db.all(`
          SELECT u.phone_number, p.credits_committed
          FROM participations p
          JOIN users u ON p.user_id = u.id
          WHERE p.event_id = ?
        `, [events[0].id], (err, participantsData) => {
          if (err) return res.status(500).send('Database error');

          res.render('home', {
            events,
            lastEvent,
            participants: participantsData.map(p => ({
              ...p,
              anonymized: anonymizePhone(p.phone_number)
            })),
            anonymizePhone
          });
        });
      } else {
        res.render('home', {
          events,
          lastEvent,
          participants: [],
          anonymizePhone
        });
      }
    });
  });
});

// LOGIN
app.post('/login', (req, res) => {
  const { phone, password } = req.body;
  
  db.get('SELECT * FROM users WHERE phone_number = ?', [phone], (err, user) => {
    if (err) return res.status(500).send('Database error');
    
    if (!user || !bcrypt.compareSync(password, user.password_hash)) {
      return res.json({ success: false, message: 'Número ou senha inválidos' });
    }

    req.session.user = {
      id: user.id,
      phone_number: user.phone_number,
      credits_balance: user.credits_balance,
      is_admin: user.is_admin
    };

    res.json({ 
      success: true, 
      redirect: user.is_admin ? '/admin' : '/usuario',
      user: req.session.user
    });
  });
});

// LOGOUT
app.get('/logout', (req, res) => {
  req.session.destroy();
  res.redirect('/');
});

// USER PAGE
app.get('/usuario', requireAuth, (req, res) => {
  db.get('SELECT * FROM users WHERE id = ?', [req.session.user.id], (err, user) => {
    if (err) return res.status(500).send('Database error');
    
    req.session.user.credits_balance = user.credits_balance;

    db.all(`
      SELECT e.*, 
             p.credits_committed as my_credits,
             (SELECT COUNT(*) FROM participations WHERE event_id = e.id) as participant_count
      FROM events e
      LEFT JOIN participations p ON e.id = p.event_id AND p.user_id = ?
      WHERE e.status = 'upcoming' AND e.event_datetime > datetime('now')
      ORDER BY e.event_datetime ASC
    `, [req.session.user.id], (err, events) => {
      if (err) return res.status(500).send('Database error');

      res.render('user', { 
        user: req.session.user,
        events,
        rechargeAmounts: [500, 1000, 5000, 10000]
      });
    });
  });
});

// JOIN EVENT
app.post('/evento/entrar', requireAuth, (req, res) => {
  const { eventId, credits } = req.body;
  const userId = req.session.user.id;
  const creditsAmount = parseInt(credits);

  db.get('SELECT credits_balance FROM users WHERE id = ?', [userId], (err, user) => {
    if (err) return res.status(500).send('Database error');
    
    if (user.credits_balance < creditsAmount) {
      return res.json({ success: false, message: 'Saldo insuficiente' });
    }

    db.get('SELECT id FROM participations WHERE user_id = ? AND event_id = ?', [userId, eventId], (err, existing) => {
      if (err) return res.status(500).send('Database error');
      
      if (existing) {
        return res.json({ success: false, message: 'Já está inscrito neste evento' });
      }

      db.get('SELECT * FROM events WHERE id = ? AND status = ?', [eventId, 'upcoming'], (err, event) => {
        if (err) return res.status(500).send('Database error');
        
        if (!event) {
          return res.json({ success: false, message: 'Evento não disponível' });
        }

        db.serialize(() => {
          db.run('UPDATE users SET credits_balance = credits_balance - ? WHERE id = ?', [creditsAmount, userId]);
          db.run('INSERT INTO participations (user_id, event_id, credits_committed) VALUES (?, ?, ?)', 
            [userId, eventId, creditsAmount]);
        });

        req.session.user.credits_balance -= creditsAmount;
        res.json({ success: true, newBalance: req.session.user.credits_balance });
      });
    });
  });
});

// LEAVE EVENT
app.post('/evento/sair', requireAuth, (req, res) => {
  const { eventId } = req.body;
  const userId = req.session.user.id;

  db.get('SELECT * FROM participations WHERE user_id = ? AND event_id = ?', [userId, eventId], (err, participation) => {
    if (err) return res.status(500).send('Database error');
    
    if (!participation) {
      return res.json({ success: false, message: 'Não está inscrito neste evento' });
    }

    db.get('SELECT * FROM events WHERE id = ? AND status = ?', [eventId, 'upcoming'], (err, event) => {
      if (err) return res.status(500).send('Database error');
      
      if (!event) {
        return res.json({ success: false, message: 'Não é possível sair deste evento' });
      }

      db.serialize(() => {
        db.run('UPDATE users SET credits_balance = credits_balance + ? WHERE id = ?', 
          [participation.credits_committed, userId]);
        db.run('DELETE FROM participations WHERE user_id = ? AND event_id = ?', [userId, eventId]);
      });

      req.session.user.credits_balance += participation.credits_committed;
      res.json({ success: true, newBalance: req.session.user.credits_balance });
    });
  });
});

// ============== ADMIN ROUTES ==============

app.get('/admin', requireAdmin, (req, res) => {
  db.all('SELECT id, phone_number, credits_balance, created_at FROM users WHERE is_admin = 0 ORDER BY created_at DESC', (err, users) => {
    if (err) return res.status(500).send('Database error');
    
    db.all(`
      SELECT e.*, 
             (SELECT COUNT(*) FROM participations WHERE event_id = e.id) as participant_count
      FROM events e 
      WHERE e.status = 'upcoming'
      ORDER BY e.event_datetime ASC
    `, (err, upcomingEvents) => {
      if (err) return res.status(500).send('Database error');
      
      db.all(`
        SELECT e.*, u.phone_number as winner_phone,
               (SELECT COUNT(*) FROM participations WHERE event_id = e.id) as participant_count
        FROM events e
        LEFT JOIN users u ON e.winner_id = u.id
        WHERE e.status = 'completed'
        ORDER BY e.event_datetime DESC
        LIMIT 20
      `, (err, pastEvents) => {
        if (err) return res.status(500).send('Database error');

        res.render('admin', {
          users,
          upcomingEvents,
          pastEvents,
          rechargeAmounts: [500, 1000, 5000, 10000]
        });
      });
    });
  });
});

// CREATE USER
app.post('/admin/usuario/criar', requireAdmin, (req, res) => {
  const { phone } = req.body;
  
  db.get('SELECT id FROM users WHERE phone_number = ?', [phone], (err, existing) => {
    if (err) return res.status(500).send('Database error');
    
    if (existing) {
      return res.json({ success: false, message: 'Utilizador já existe' });
    }

    const password = generatePassword();
    const hashedPassword = bcrypt.hashSync(password, 10);

    db.run('INSERT INTO users (phone_number, password_hash, credits_balance) VALUES (?, ?, 0)', 
      [phone, hashedPassword], function(err) {
      if (err) return res.status(500).send('Database error');
      
      res.json({ success: true, phone, password });
    });
  });
});

// RECHARGE USER
app.post('/admin/usuario/recarregar', requireAdmin, (req, res) => {
  const { phone, amount } = req.body;
  const creditsAmount = parseInt(amount);

  db.get('SELECT id FROM users WHERE phone_number = ?', [phone], (err, user) => {
    if (err) return res.status(500).send('Database error');
    
    if (!user) {
      return res.json({ success: false, message: 'Utilizador não encontrado' });
    }

    db.run('UPDATE users SET credits_balance = credits_balance + ? WHERE id = ?', 
      [creditsAmount, user.id], function(err) {
      if (err) return res.status(500).send('Database error');
      
      db.get('SELECT credits_balance FROM users WHERE id = ?', [user.id], (err, updatedUser) => {
        if (err) return res.status(500).send('Database error');
        
        res.json({ success: true, newBalance: updatedUser.credits_balance });
      });
    });
  });
});

// CREATE EVENT
app.post('/admin/evento/criar', requireAdmin, (req, res) => {
  const { name, datetime, prizeType, prizeValue } = req.body;

  db.run('INSERT INTO events (name, event_datetime, prize_type, prize_value) VALUES (?, ?, ?, ?)',
    [name, datetime, prizeType, prizeValue], function(err) {
    if (err) return res.status(500).send('Database error');
    
    res.json({ success: true });
  });
});

// DELETE EVENT
app.post('/admin/evento/eliminar', requireAdmin, (req, res) => {
  const { eventId } = req.body;

  db.all('SELECT user_id, credits_committed FROM participations WHERE event_id = ?', [eventId], (err, participations) => {
    if (err) return res.status(500).send('Database error');
    
    db.serialize(() => {
      for (const p of participations) {
        db.run('UPDATE users SET credits_balance = credits_balance + ? WHERE id = ?', 
          [p.credits_committed, p.user_id]);
      }

      db.run('DELETE FROM participations WHERE event_id = ?', [eventId]);
      db.run('DELETE FROM events WHERE id = ?', [eventId]);
    });

    res.json({ success: true });
  });
});

// RUN ROULETTE
app.post('/admin/evento/rodar', requireAdmin, (req, res) => {
  const { eventId } = req.body;

  db.all(`
    SELECT p.*, u.phone_number
    FROM participations p
    JOIN users u ON p.user_id = u.id
    WHERE p.event_id = ?
  `, [eventId], (err, participants) => {
    if (err) return res.status(500).send('Database error');
    
    if (participants.length === 0) {
      return res.json({ success: false, message: 'Nenhum participante neste evento' });
    }

    const totalCredits = participants.reduce((sum, p) => sum + p.credits_committed, 0);
    let random = Math.random() * totalCredits;
    let winner = participants[0];

    for (const p of participants) {
      random -= p.credits_committed;
      if (random <= 0) {
        winner = p;
        break;
      }
    }

    db.run('UPDATE events SET status = ?, winner_id = ? WHERE id = ?',
      ['completed', winner.user_id, eventId], function(err) {
      if (err) return res.status(500).send('Database error');
      
      db.get('SELECT * FROM events WHERE id = ?', [eventId], (err, event) => {
        if (err) return res.status(500).send('Database error');

        res.json({ 
          success: true, 
          winner: {
            id: winner.user_id,
            phone: winner.phone_number,
            anonymized: anonymizePhone(winner.phone_number)
          },
          event,
          participants: participants.map(p => ({
            ...p,
            anonymized: anonymizePhone(p.phone_number)
          }))
        });
      });
    });
  });
});

// API ENDPOINTS
app.get('/api/evento/:id', (req, res) => {
  db.get(`
    SELECT e.*, u.phone_number as winner_phone,
           (SELECT COUNT(*) FROM participations WHERE event_id = e.id) as participant_count
    FROM events e
    LEFT JOIN users u ON e.winner_id = u.id
    WHERE e.id = ?
  `, [req.params.id], (err, event) => {
    if (err) return res.status(500).send('Database error');
    
    if (!event) {
      return res.json({ success: false });
    }

    db.all(`
      SELECT u.phone_number
      FROM participations p
      JOIN users u ON p.user_id = u.id
      WHERE p.event_id = ?
    `, [req.params.id], (err, participants) => {
      if (err) return res.status(500).send('Database error');

      res.json({
        success: true,
        event,
        participants: participants.map(p => ({
          phone: p.phone_number,
          anonymized: anonymizePhone(p.phone_number)
        })),
        winnerAnonymized: event.winner_phone ? anonymizePhone(event.winner_phone) : null
      });
    });
  });
});

app.get('/api/eventos', (req, res) => {
  db.all(`
    SELECT e.*, 
           (SELECT COUNT(*) FROM participations WHERE event_id = e.id) as participant_count
    FROM events e 
    WHERE e.status = 'upcoming' AND e.event_datetime > datetime('now')
    ORDER BY e.event_datetime ASC 
    LIMIT 2
  `, (err, events) => {
    if (err) return res.status(500).send('Database error');
    
    res.json({ success: true, events });
  });
});

function listenOnNextAvailablePort(port) {
  const server = app.listen(port, () => {
    const actualPort = server.address().port;
    console.log(`🎰 Baicaa rodando em http://localhost:${actualPort}`);
  });

  server.on('error', (err) => {
    if (err && err.code === 'EADDRINUSE') {
      server.close(() => listenOnNextAvailablePort(port + 1));
      return;
    }

    console.error('Server error:', err);
    process.exit(1);
  });
}

listenOnNextAvailablePort(START_PORT);
